<?php

namespace App\Livewire\Components;

use Illuminate\Support\Facades\DB;
use Intervention\Image\Image;
use Livewire\Attributes\On;
use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\Photo;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Livewire\TemporaryUploadedFile;


class FileUpload extends Component
{
    use WithFileUploads;

    public $type;
    public $isMultiple = false;
    public $label;
    public $acceptedFormats;
    public $maxSize;
    public $dimensions;
    public $modelId;
    public $modelClass;
    public $wireModel;
    public $required = false;
    public $data;
    public $uniqueKey;
    public $uploadDisk = 'livewire_file';
    public $pathPrefix = 'assets/admin/uploads';
    public $maxFiles = 6;
    public $files = [];
    public $previewUrls = [];
    public $applyWatermark = false;

    public function mount(
        $type,
        $isMultiple = false,
        $label = '',
        $acceptedFormats = '',
        $maxSize = null,
        $dimensions = null,
        $modelId = null,
        $modelClass = null,
        $wireModel = '',
        $required = false,
        $data = null,
        $uniqueKey = '',
        $uploadDisk = 'livewire_file',
        $pathPrefix = 'assets/admin/uploads',
        $maxFiles = 6,
        $applyWatermark = false
    ) {
        $this->type = $type;
        $this->isMultiple = $isMultiple;
        $this->label = $label;
        $this->acceptedFormats = $acceptedFormats;
        $this->maxSize = $maxSize;
        $this->dimensions = $dimensions;
        $this->modelId = $modelId;
        $this->modelClass = $modelClass;
        $this->wireModel = $wireModel;
        $this->required = $required;
        $this->data = $data;
        $this->uniqueKey = $uniqueKey;
        $this->uploadDisk = $uploadDisk;
        $this->pathPrefix = $pathPrefix;
        $this->maxFiles = $isMultiple ? $maxFiles : 1;
        $this->applyWatermark = $type === 'gallery' ? $applyWatermark : false;

        $this->loadPreviews();
    }

    protected function loadPreviews()
    {
        $this->previewUrls = [];
        if ($this->data) {
            $data = $this->isMultiple ? collect($this->data) : collect([$this->data]);
            $this->previewUrls = $data->filter()->map(function ($file) {
                $extension = pathinfo($file['path'], PATHINFO_EXTENSION);
                $type = in_array(strtolower($extension), ['jpg', 'jpeg', 'png', 'webp', 'gif']) ? 'image' :
                    (strtolower($extension) === 'mp4' ? 'video' : 'pdf');
                return [
                    'id' => $file['id'],
                    'path' => url($file['path']),
                    'name' => basename($file['path']),
                    'size' => Storage::disk($this->uploadDisk)->exists($file['path']) ? Storage::disk($this->uploadDisk)->size($file['path']) : 0,
                    'type' => $type,
                ];
            })->values()->toArray();
        }
    }

    public function updatedFiles($value)
    {
        if (!$value) {
            $this->files = [];
            return;
        }

        $this->validateFiles();
    }

    public function validateFiles()
    {
        $rules = [
            'files.*' => [
                $this->required && empty($this->data) ? 'required' : 'nullable',
                'file',
                'mimes:' . $this->acceptedFormats,
                $this->maxSize ? 'max:' . $this->maxSize : '',
            ],
        ];

        if ($this->isMultiple) {
            $rules['files'] = [
                'max:' . $this->maxFiles,
                function ($attribute, $value, $fail) {
                    $existingFilesCount = count($this->previewUrls);
                    $newFilesCount = count($value);
                    $totalFiles = $existingFilesCount + $newFilesCount;
                    if ($totalFiles > $this->maxFiles) {
                        $fail(translateText("Maximum :maxFiles files allowed, including existing files.", lang_admin(), ['maxFiles' => $this->maxFiles]));
                    }
                },
            ];
        }

        if ($this->dimensions) {
            [$width, $height] = explode('x', $this->dimensions);
            $rules['files.*'][] = function ($attribute, $value, $fail) use ($width, $height) {
                if ($value instanceof TemporaryUploadedFile && in_array(strtolower($value->getClientOriginalExtension()), ['jpg', 'jpeg', 'png', 'webp', 'gif'])) {
                    [$imgWidth, $imgHeight] = getimagesize($value->getRealPath());
                    if ($imgWidth != $width || $imgHeight != $height) {
                        $fail(translateText('The :attribute dimensions must be :dimensions.', lang_admin(), ['dimensions' => $this->dimensions]));
                    }
                }
            };
        }

        $validator = Validator::make(['files' => $this->files], $rules);
        if ($validator->fails()) {
            $this->files = [];
            $this->dispatch('file-upload-failed', type: $this->type, error: implode('<br>', $validator->errors()->all()));
        }
    }

    public function deleteFile($fileId)
    {
        $photo = Photo::find($fileId);
        if ($photo && Storage::disk($this->uploadDisk)->exists($photo->path)) {
            Storage::disk($this->uploadDisk)->delete($photo->path);
            $photo->delete();
            $this->data = $this->isMultiple
                ? collect($this->data)->filter(fn($item) => $item->id != $fileId)->values()->all()
                : null;
            $this->loadPreviews();
            $this->dispatch('previews-updated', previews: $this->previewUrls);
            $this->dispatch('alert',
                icon: 'info',
                title: translateText('Deleted', lang_admin()),
                text: translateText('File deleted successfully.', lang_admin())
            );
        } else {
            $this->dispatch('file-upload-failed',
                type: $this->type,
                error: translateText('File not found.', lang_admin())
            );
        }
    }

    #[On('save-files')]
    public function saveFiles($id)
    {
        try {
            if (!$this->files) {
                return;
            }

            DB::beginTransaction();

            $method = $this->type;
            $modelClass = $this->modelClass;
            $model = $modelClass::findOrFail($id);

            if ($this->modelId && !$this->isMultiple) {
                $existingPhotos = $model->$method()->where('type', $this->type)->get();
                foreach ($existingPhotos as $photo) {
                    if (Storage::disk($this->uploadDisk)->exists($photo->path)) {
                        Storage::disk($this->uploadDisk)->delete($photo->path);
                    }
                    $photo->delete();
                }
            }

            $this->validateFiles();

            $existingFilesCount = $this->isMultiple ? $model->$method()->where('type', $this->type)->count() : 0;
            $newFilesCount = count($this->files);
            if ($this->isMultiple && ($existingFilesCount + $newFilesCount) > $this->maxFiles) {
                DB::rollBack();
                $this->files = [];
                $this->dispatch('file-upload-failed', type: $this->type, error: translateText("Cannot upload. Maximum :maxFiles files allowed, including existing files.", lang_admin(), ['maxFiles' => $this->maxFiles]));
                return;
            }

            foreach ($this->files as $file) {
                $relativePath = $this->pathPrefix . '/' . $this->type . '/' . my_jdate(date('Y/m/d'), 'Y-m-d') . '/' . time();
                $originalName = $file->getClientOriginalName();
                $fullPath = $relativePath . '/' . $originalName;

                if ($this->type === 'gallery' && $this->applyWatermark && in_array(strtolower($file->getClientOriginalExtension()), ['jpg', 'jpeg', 'png', 'webp', 'gif'])) {
                    $image = Image::make($file->getRealPath());
                    $image->text(' Watermark', 20, 20, function ($font) {
                        $font->file(url('assets/front/fonts/fa-brands-400.ttf'));
                        $font->size(24);
                        $font->color('#ffffff');
                        $font->align('left');
                        $font->valign('top');
                        $font->angle(0);
                    });
                    Storage::disk($this->uploadDisk)->put($fullPath, $image->encode());
                } else {
                    $file->storeAs($relativePath, $originalName, $this->uploadDisk);
                }

                $photo = new Photo();
                $photo->type = $this->type;
                $photo->path = $fullPath;
                $photo->status = 'active';
                $model->$method()->save($photo);
            }

            DB::commit();

            $this->files = [];
            $this->data = $this->isMultiple ? $model->$method()->where('type', $this->type)->get()->toArray() : $model->$method()->where('type', $this->type)->first();
            $this->loadPreviews();
            $this->dispatch('file-upload-ready');
            $this->dispatch('alert', icon: 'success', title: translateText('Success', lang_admin()), text: translateText('Files saved successfully.', lang_admin()));
        } catch (\Exception $e) {
            DB::rollBack();
            if (!$this->modelId) {
                $model->delete();
            }
            $this->files = [];
            $this->dispatch('file-upload-failed', type: $this->type, error: translateText('Failed to save files: ', lang_admin()) . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.components.file-upload', [
            'previewUrls' => $this->previewUrls,
        ]);
    }
}