<?php

namespace App\Livewire\Dashboard\Company\CompanyCertificate;

use Illuminate\Support\Facades\DB;
use Livewire\Component;
use App\Models\CompanyCertificate;

class Create extends Component
{
    public $userId;
    public $sort = [];
    public $data = [
        'certificate_title' => null,
        'certificate_id_number' => null,
        'issued_by' => null,
        'issued_date' => null,
        'expiry_date' => null,
        'certificate_description' => null,
        'status' => null,
    ];

    public function mount($userId)
    {
        $this->userId = $userId;
    }

    public function save()
    {
        try {
            $this->validate();
            DB::beginTransaction();
            $maxSort = CompanyCertificate::where('user_id', $this->userId)->max('sort');
            $nextSort = ($maxSort !== null) ? $maxSort + 1 : 1;
            $item = CompanyCertificate::create([
                'user_id' => $this->userId,
                'certificate_title' => $this->data['certificate_title'],
                'certificate_id_number' => $this->data['certificate_id_number'],
                'issued_by' => $this->data['issued_by'],
                'issued_date' => $this->data['issued_date'],
                'expiry_date' => $this->data['expiry_date'],
                'certificate_description' => $this->data['certificate_description'],
                'status' => $this->data['status'] == '1' ? 'active' : 'pending',
                'sort' => $nextSort,
            ]);
            $this->dispatch('save-files', id: $item->id);
            DB::commit();
            $this->dispatch('itemRender');
            $this->dispatch('insert', icon: 'success', title: 'Success', text: 'Certificate added successfully.');
            $this->reset('data');
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            $messages = $e->validator->errors()->all();
            $errorMessage = implode('<br>', array_map(function ($error) {
                $error = str_replace('data.', '', $error);
                return "- $error";
            }, $messages));
            $this->dispatch('errorShow', icon: 'warning', title: 'Validation Error', text: $errorMessage);
        } catch (\Exception $e) {
            DB::rollBack();
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.dashboard.company.company-certificate.create');
    }

    protected function rules()
    {
        return [
            'data.status' => 'nullable|in:1,active,pending',
            'data.certificate_title' => 'required|string|max:255',
            'data.certificate_id_number' => 'required|string|max:100|unique:company_certificates,certificate_id_number',
            'data.issued_by' => 'required|string|max:255',
            'data.issued_date' => 'required|date',
            'data.expiry_date' => 'nullable|date|after:data.issued_date',
            'data.certificate_description' => 'nullable|string',
        ];
    }
}
