<?php

namespace App\Livewire\Dashboard\Company\CompanyCertificate;

use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Layout;
use Livewire\Attributes\On;
use Livewire\Component;
use App\Models\CompanyCertificate;

class Edit extends Component
{
    public $item;
    public $userId;
    public $baseFields =[
        'certificate_title',
        'certificate_id_number',
        'issued_by',
        'issued_date',
        'expiry_date',
        'certificate_description',
        'status',
        'photo1',
        'photo2',
        'photo3',
    ];
    public $data = [];

    public function mount($id)
    {
        $this->item = CompanyCertificate::with(['photo1', 'photo2', 'photo3'])
            ->where('id', $id)
            ->first();
        $this->loadData();
    }

    protected function loadData()
    {
        try {
            foreach ($this->baseFields as $field) {
                $this->data[$field] = $this->item->$field;
            }
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: 'Failed to load form data.');
        }
    }


    public function save()
    {
        try {
            $this->validate();
            DB::beginTransaction();
            $this->item->update([
                'certificate_title' => $this->data['certificate_title'],
                'certificate_id_number' => $this->data['certificate_id_number'],
                'issued_by' => $this->data['issued_by'],
                'issued_date' => $this->data['issued_date'],
                'expiry_date' => $this->data['expiry_date'],
                'certificate_description' => $this->data['certificate_description'],
                'status' => ($this->data['status'] == '1' || $this->data['status'] == 'active') ? 'active' : 'pending',
            ]);
            $this->dispatch('save-files', id: $this->item->id);
            DB::commit();
            $this->dispatch('insert', icon: 'success', title: 'Success', text: translateText('The form was submitted successfully.', lang_admin()));
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            $messages = $e->validator->errors()->all();
            $errorMessage = implode('<br>', array_map(function ($error) {
                $error = str_replace('data.', '', $error);
                return "- $error";
            }, $messages));
            $this->dispatch('errorShow', icon: 'warning', title: '! Validation Error !', text: $errorMessage);
        } catch (\Exception $e) {
            DB::rollBack();
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: $e->getMessage());
        }
    }

    #[Layout('components.layouts.app', [
        'assets' => ['editor'],
        'theme' => '7'
    ])]
    public function render()
    {
        return view('livewire.dashboard.company.company-certificate.edit', [
            'title' => 'Edit Company Certificate',
            'page_title' => ['Company', 'Company Certificate'],
        ]);
    }

    protected function rules()
    {
        return [
            'data.status' => 'nullable|in:1,active,pending',
            'data.certificate_title' => 'required|string|max:255',
            'data.certificate_id_number' => 'required|string|max:100|',
            'data.issued_by' => 'required|string|max:255',
            'data.issued_date' => 'required|date',
            'data.expiry_date' => 'nullable|date|after:data.issued_date',
            'data.certificate_description' => 'nullable|string',
        ];
    }
}