<?php

namespace App\Livewire\Dashboard\Company\CompanyFaq;

use Illuminate\Support\Facades\DB;
use Livewire\Component;
use App\Models\CompanyFaq;

class Create extends Component
{
    public $userId;
    public $sort = [];
    public $translatableFields = [
        'question',
        'answer',
        'status',
    ];
    public $data = [];

    public function mount($userId)
    {
        $this->userId = $userId;
    }

    protected function loadData()
    {
        foreach (tab_langs() as $lang) {
            $suffix = $lang->status == 'default' ? '' : '_' . $lang->lang;
            foreach ($this->translatableFields as $field) {
                $dbField = $field . $suffix;
                $this->data[$dbField] = null;
            }
        }
    }

    public function save()
    {

        try {
            $this->validate();
            if (!array_key_exists('status', $this->data)) {
                $this->data['status'] = 'pending';
            }
            DB::beginTransaction();
            $maxSort = CompanyFaq::where('user_id', $this->userId)->max('sort');
            $nextSort = ($maxSort !== null) ? $maxSort + 1 : 1;
            foreach (tab_langs() as $lang) {
                $suffix = $lang->status == 'default' ? '' : '_' . $lang->lang;
                foreach ($this->data as $key => $value) {
                    if (str_starts_with($key, 'status')) {
                        $dbField = 'status' . $suffix;
                        $this->data[$dbField] = $value == '1' ? 'active' : 'pending';
                    }
                }
            }

            $item = CompanyFaq::create([
                'user_id' => $this->userId,
                'question' => $this->data['question'],
                'answer' => $this->data['answer'],
                'status' => $this->data['status'],
                'sort' => $nextSort,
            ]);
            store_lang_for_array_data($item, $this->data, ['question', 'answer','status']);
            DB::commit();
            $this->dispatch('itemRender');
            $this->dispatch('insert', icon: 'success', title: 'Success', text: 'Certificate added successfully.');
            $this->reset('data');
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            $messages = $e->validator->errors()->all();
            $errorMessage = implode('<br>', array_map(function ($error) {
                $error = str_replace('data.', '', $error);
                return "- $error";
            }, $messages));
            $this->dispatch('errorShow', icon: 'warning', title: 'Validation Error', text: $errorMessage);
        } catch (\Exception $e) {
            DB::rollBack();
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.dashboard.company.company-faq.create');
    }

    protected function rules()
    {
        $rules = [
        ];
        foreach (tab_langs() as $lang) {
            $suffix = $lang->status == 'default' ? '' : '_' . $lang->lang;
            $rules["data.question{$suffix}"] = $lang->status == 'default' ? 'required' : 'nullable';
            $rules["data.answer{$suffix}"] = $lang->status == 'default' ? 'required' : 'nullable';
            $rules["data.status{$suffix}"] = 'nullable|in:1,active,pending';
        }

        return $rules;
    }
}
