<?php

namespace App\Livewire\Dashboard\Company;

use App\Models\Photo;
use App\Models\ContactInformation;
use App\Models\Country;
use App\Models\State;
use App\Models\City;
use App\Models\CompanyForm;
use App\Services\DataCacheService;
use Livewire\Attributes\Layout;
use Livewire\Attributes\On;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class ContactInformationForm extends Component
{
    use WithFileUploads;

    public $baseFields = [
        'email_address',
        'backup_email',
        'country_code1',
        'city_code',
        'phone_number',
        'country_code2',
        'mobile_number',
        'country_code3',
        'city_code2',
        'fax_number',
        'supported_languages',
        'social_media',
        'country_id',
        'state_id',
        'city_id',
        'street',
        'post_code',
        'time_zone',
        'profile',
    ];
    public $translatableFields = [
        'contact_person',
        'job_title',
        'department',
    ];
    public $country_code = [];
    public $countries = [];
    public $states = [];
    public $cities = [];
    public $langS = [];
    public $social_media_types = [];
    public $time_zone = [];
    public $item;
    public $data = [];
    public $social_media = [];

    public function mount($id)
    {
        try {
            $this->item = ContactInformation::firstOrCreate(['user_id' => $id], ['user_id' => $id]);
            $this->loadData();
            $this->country_code = Country::orderBy('name')
                ->get(['phonecode', 'name'])
                ->mapWithKeys(function ($item) {
                    return [$item->phonecode => "{$item->name} (+{$item->phonecode})"];
                })
                ->all();
            $this->countries = DataCacheService::getCountries();
            $this->time_zone = DataCacheService::getTimeZones();
            $this->langS = CompanyForm::where('type', 'lang')->pluck('title', 'title')->all();
            $this->social_media_types = CompanyForm::where('type', 'social_media')->pluck('title', 'title')->all();
            $this->social_media = json_decode($this->item->social_media, true) ?? [];
            $this->loadStatesAndCities();
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: 'Failed to initialize form.');
        }
    }

    protected function loadData()
    {
        try {
            foreach ($this->baseFields as $field) {
                if ($field == 'supported_languages') {
                    $this->data[$field] = json_decode($this->item->$field, true) ?? [];
                } elseif ($field == 'social_media') {
                    $this->data[$field] = json_decode($this->item->$field, true) ?? [];
                } elseif ($field == 'profile') {
                    $this->data[$field] = $this->item->$field()->first();
                } else {
                    $this->data[$field] = $this->item->$field;
                }
            }
            foreach (tab_langs() as $lang) {
                $suffix = $lang->status == 'default' ? '' : '_' . $lang->lang;
                foreach ($this->translatableFields as $field) {
                    $dbField = $field . $suffix;
                    $this->data[$dbField] = read_lang($this->item, $field, $lang->lang);
                }
            }
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: 'Failed to load form data.');
        }
    }

    protected function loadStatesAndCities()
    {
        try {
            $this->states = $this->data['country_id'] ? State::where('country_id', $this->data['country_id'])->pluck('name', 'id')->all() : [];
            $this->cities = $this->data['state_id'] ? City::where('state_id', $this->data['state_id'])->pluck('name', 'id')->all() : [];
            $this->dispatch('update-options-state_id', options: $this->states);
            $this->dispatch('update-options-city_id', options: $this->cities);
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: 'Failed to load states and cities.');
        }
    }

    #[On('tomselect-updated')]
    public function handleTomSelectUpdated($field, $value)
    {
        try {
            $this->data[$field] = $value;
            if ($field === 'country_id') {
                $this->data['state_id'] = null;
                $this->data['city_id'] = null;
                $this->states = $value ? State::where('country_id', $value)->pluck('name', 'id')->all() : [];
                $this->cities = [];
                $this->dispatch('update-options-state_id', options: $this->states);
                $this->dispatch('update-options-city_id', options: $this->cities);
            } elseif ($field === 'state_id') {
                $this->data['city_id'] = null;
                $this->cities = $value ? City::where('state_id', $value)->pluck('name', 'id')->all() : [];
                $this->dispatch('update-options-city_id', options: $this->cities);
            }
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: 'Failed to handle TomSelect update.');
        }
    }

    public function save()
    {
        try {

            $this->validate();

            $this->item->update([
                'contact_person' => $this->data['contact_person'],
                'job_title' => $this->data['job_title'],
                'department' => $this->data['department'],
                'email_address' => $this->data['email_address'],
                'backup_email' => $this->data['backup_email'],
                'country_code1' => $this->data['country_code1'],
                'city_code' => $this->data['city_code'],
                'phone_number' => $this->data['phone_number'],
                'country_code2' => $this->data['country_code2'],
                'mobile_number' => $this->data['mobile_number'],
                'country_code3' => $this->data['country_code3'],
                'city_code2' => $this->data['city_code2'],
                'fax_number' => $this->data['fax_number'],
                'supported_languages' => json_encode($this->data['supported_languages']),
                'social_media' => json_encode($this->data['social_media']),
                'country_id' => $this->data['country_id'],
                'state_id' => $this->data['state_id'],
                'city_id' => $this->data['city_id'],
                'street' => $this->data['street'],
                'post_code' => $this->data['post_code'],
                'time_zone' => $this->data['time_zone'],
            ]);


            $this->dispatch('save-files', id: $this->item->id);

            store_lang_for_array_data($this->item, $this->data, ['contact_person', 'job_title', 'department'], 'edit');
            $this->dispatch('insert', icon: 'success', title: 'Success', text: translateText('The form was submitted successfully.', lang_admin()));
            $this->loadData();
        } catch (\Illuminate\Validation\ValidationException $e) {
            $messages = $e->validator->errors()->all();
            $errorMessage = implode('<br>', array_map(function ($error) {
                $error = str_replace('data.', '', $error);
                return "- $error";
            }, $messages));
            $this->dispatch('errorShow', icon: 'warning', title: '! Validation Error !', text: $errorMessage);
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: $e->getMessage());
        }
    }

    #[Layout('components.layouts.app', [
        'assets' => ['editor', 'tom-select','country-manage'],
        'theme' => '7'
    ])]
    public function render()
    {
        return view('livewire.dashboard.company.contact-information-form', [
            'title' => 'Edit contact information',
            'page_title' => ['Company', 'Contact Information'],
        ]);
    }

    protected function rules()
    {
        $rules = [
            'data.contact_person' => 'required|string|max:255',
            'data.job_title' => 'required|string|max:255',
            'data.department' => 'nullable|string|max:255',
            'data.email_address' => 'required|email|max:255',
            'data.backup_email' => 'required|email|max:255',
            'data.phone_number' => 'required|string|max:20',
            'data.mobile_number' => 'required|string|max:20',
            'data.fax_number' => 'nullable|string|max:20',
            'data.supported_languages' => 'required|array|min:1',
            'data.social_media' => 'required|array|min:1',
            'data.country_id' => 'required|exists:countries,id',
            'data.state_id' => 'required|exists:states,id',
            'data.city_id' => 'required|exists:cities,id',
            'data.street' => 'required|string|max:255',
            'data.post_code' => 'required|string|max:20',
            'data.time_zone' => 'required|string|max:50',
            'data.social_media.*.type' => 'required|string',
            'data.social_media.*.value' => 'required|string|max:255',
        ];

        foreach (tab_langs() as $lang) {
            $suffix = $lang->status == 'default' ? '' : '_' . $lang->lang;
            $rules["data.contact_person{$suffix}"] = $lang->status == 'default' ? 'required|string|max:255' : 'nullable|string|max:255';
            $rules["data.job_title{$suffix}"] = $lang->status == 'default' ? 'required|string|max:255' : 'nullable|string|max:255';
            $rules["data.department{$suffix}"] = 'nullable|string|max:255';
        }

        return $rules;
    }

}