<?php

namespace App\Livewire\Dashboard\Company;

use App\Models\FactoryInformation;
use App\Models\Photo;
use Livewire\Attributes\Layout;
use Livewire\Attributes\On;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\DB;

class FactoryInformationForm extends Component
{
    use WithFileUploads;

    public $baseFields = [
        'factory_location',
        'factory_total_area',
        'no_production_staff',
        'no_qc_staff',
        'no_rd_staff',
        'no_production_lines',
        'annual_output',
        'average_lead_time',
        'video',
        'gallery',
        'video_url',
    ];
    public $item;
    public $data = [];

    public function mount($id)
    {
        $this->item = FactoryInformation::with('gallery', 'video')->firstOrCreate(['user_id' => $id], ['user_id' => $id]);
        $this->loadData();
    }

    protected function loadData()
    {
        foreach ($this->baseFields as $field) {
            if ($field == 'gallery') {
                $this->data[$field] = $this->item->gallery;
            } elseif ($field == 'video') {
                $this->data[$field] = $this->item->video()->first();
            } else {
                $this->data[$field] = $this->item->$field;
            }
        }
    }

    public function save()
    {
        try {
            $this->validate();

            DB::beginTransaction();

            $this->item->update([
                'factory_location' => $this->data['factory_location'],
                'factory_total_area' => $this->data['factory_total_area'],
                'no_production_staff' => $this->data['no_production_staff'],
                'no_qc_staff' => $this->data['no_qc_staff'],
                'no_rd_staff' => $this->data['no_rd_staff'],
                'no_production_lines' => $this->data['no_production_lines'],
                'annual_output' => $this->data['annual_output'],
                'average_lead_time' => $this->data['average_lead_time'],
                'video_url' => $this->data['video_url'],
            ]);

            $this->dispatch('save-files', id: $this->item->id);

            DB::commit();

            $this->dispatch('insert', icon: 'success', title: 'Success', text: translateText('The form was submitted successfully.', lang_admin()));
            $this->loadData();
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            if (!$this->item->exists) {
                $this->item->delete();
            }
            $messages = $e->validator->errors()->all();
            $errorMessage = implode('<br>', array_map(function ($error) {
                $error = str_replace('data.', '', $error);
                return "- $error";
            }, $messages));
            $this->dispatch('errorShow', icon: 'warning', title: '! Validation Error !', text: $errorMessage);
        } catch (\Exception $e) {
            DB::rollBack();
            if (!$this->item->exists) {
                $this->item->delete();
            }
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: translateText('An unexpected error occurred.', lang_admin()));
        }
    }

    #[Layout('components.layouts.app', [
        'assets' => ['file-upload'],
        'theme' => '7'
    ])]
    public function render()
    {
        return view('livewire.dashboard.company.factory-information-form', [
            'title' => 'Edit Factory Information',
            'page_title' => ['Company', 'Factory Information'],
        ]);
    }

    protected function rules()
    {
        return [
            'data.factory_location' => 'nullable|string|max:255',
            'data.factory_total_area' => 'nullable|string|max:255',
            'data.no_production_staff' => 'nullable|string|max:255',
            'data.no_qc_staff' => 'nullable|string|max:255',
            'data.no_rd_staff' => 'nullable|string|max:255',
            'data.no_production_lines' => 'nullable|integer|min:0',
            'data.annual_output' => 'nullable|string|max:255',
            'data.average_lead_time' => 'nullable|string|max:255',
            'data.video_url' => 'nullable|url|max:255',
        ];
    }
}