<?php

namespace App\Livewire\Dashboard\Company;

use App\Models\Photo;
use App\Models\GeneralInformation;
use App\Models\Country;
use App\Models\CompanyForm;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Layout;
use Livewire\Attributes\On;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\DB;

class GeneralInformationForm extends Component
{
    use WithFileUploads;

    public $baseFields = [
        'year_of_established',
        'number_of_employees',
        'website',
        'accepted_currencies',
        'payment_terms',
        'oem_services',
        'working_days',
        'trademark',
        'main_markets',
        'logo',
        'banner1',
        'banner2',
        'brochure',
        'video',
        'gallery'
    ];
    public $translatableFields = [
        'company_name',
        'company_ceo',
        'company_introduction',
        'company_description'
    ];
    public $tools;

    public $countries = [];
    public $currencies;
    public $payments;
    public $days;

    public $employeesRange;
    public $item;
    public $data = [];
    public function someMethod()
    {
        return $this->tools;
    }
    public function mount($id)
    {
        $this->item = GeneralInformation::with('logo', 'banner1', 'banner2', 'gallery', 'video', 'brochure', 'countries')->firstOrCreate(['user_id' => $id], ['user_id' => $id]);
        $this->loadData();
        $this->countries = Country::query()
            ->select('id', 'name', 'emoji','region')
            ->get()
            ->mapWithKeys(function ($country) {
                return [$country->id => "{$country->emoji} {$country->name} ({$country->region})"];
            })
            ->all();
        $formData = $this->companyFormData;
        $this->currencies = $formData['currencies'];
        $this->payments = $formData['payments'];
        $this->days = $formData['days'];
        $this->employeesRange = $formData['employeesRange'];
    }

    #[Computed]
    public function companyFormData()
    {
        $data = CompanyForm::whereIn('type', [
            'currencies',
            'payment_terms',
            'days',
            'number_of_emploees'
        ])
            ->get()
            ->groupBy('type');

        return [
            'currencies' => $data->get('currencies', collect())->pluck('title', 'title')->all(),
            'payments' => $data->get('payment_terms', collect())->pluck('title', 'title')->all(),
            'days' => $data->get('days', collect())->pluck('title', 'title')->all(),
            'employeesRange' => $data->get('number_of_emploees', collect())->pluck('title', 'title')->all()
        ];
    }

    protected function loadData()
    {
        foreach ($this->baseFields as $field) {
            if ($field == 'main_markets') {
                $this->data[$field] = $this->item->mainMarkets->pluck('country_id')->all();
            } elseif (in_array($field, ['accepted_currencies', 'working_days', 'payment_terms'])) {
                $this->data[$field] = $this->terms($this->item->$field);
            } elseif ($field == 'gallery') {
                $this->data[$field] = $this->item->gallery;
            } elseif (in_array($field, ['brochure', 'video', 'logo', 'banner1', 'banner2'])) {
                $this->data[$field] = $this->item->$field()->first();
            } else {
                $this->data[$field] = $this->item->$field;
            }
        }
        foreach (tab_langs() as $lang) {
            $suffix = $lang->status == 'default' ? '' : '_' . $lang->lang;
            foreach ($this->translatableFields as $field) {
                $dbField = $field . $suffix;
                $this->data[$dbField] = read_lang($this->item, $field, $lang->lang);
            }
        }
    }

    #[On('tomselect-updated')]
    public function handleTomSelectUpdated($field, $value)
    {
        try {
            $this->data[$field] = $value;
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: 'Failed to handle TomSelect update.');
        }
    }

    #[On('ckeditor-updated')]
    public function handleCKEditorUpdated($field, $value)
    {
        try {
            $this->data[$field] = $value;
            \Log::debug('CKEditor updated:', ['field' => $field, 'value' => $value]);
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: 'Failed to handle CKEditor update.');
        }
    }

    protected function terms($item)
    {
        $termsString = $item ?? '';
        $termsArray = array_filter(array_map('trim', explode(',', $termsString)));
        return $termsArray;
    }

    public function save()
    {
        $this->dispatch('update-ckeditor-values');
        usleep(300000);
        $this->dispatch('final');
    }

    #[On('final')]
    public function final()
    {
        try {
            $this->validate();

            DB::beginTransaction();

            $this->item->update([
                'company_name' => $this->data['company_name'],
                'company_ceo' => $this->data['company_ceo'],
                'company_introduction' => $this->data['company_introduction'],
                'company_description' => $this->data['company_description'],
                'year_of_established' => $this->data['year_of_established'],
                'number_of_employees' => $this->data['number_of_employees'],
                'website' => $this->data['website'],
                'trademark' => $this->data['trademark'],
                'accepted_currencies' => implode(', ', $this->data['accepted_currencies']),
                'payment_terms' => implode(', ', $this->data['payment_terms']),
                'oem_services' => ($this->data['oem_services'] == '1' || $this->data['oem_services'] == 'yes') ? 'yes' : 'no',
                'working_days' => implode(', ', $this->data['working_days']),
            ]);

            $this->item->countries()->sync($this->data['main_markets']);

            $this->dispatch('save-files', id: $this->item->id);

            store_lang_for_array_data($this->item, $this->data, ['company_name', 'company_ceo', 'company_introduction', 'company_description'], 'edit');

            DB::commit();

            $this->dispatch('insert', icon: 'success', title: 'success', text: translateText('The form was submitted successfully.', lang_admin()));
            $this->loadData();

        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            if (!$this->item->exists) {
                $this->item->delete();
            }
            $messages = $e->validator->errors()->all();
            $errorMessage = implode('<br>', array_map(function ($error) {
                $error = str_replace('data.', '', $error);
                return "- $error";
            }, $messages));
            $this->dispatch('errorShow', icon: 'warning', title: '! Validation Error !', text: $errorMessage);
        } catch (\Exception $e) {
            DB::rollBack();
            if (!$this->item->exists) {
                $this->item->delete();
            }
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: translateText('An unexpected error occurred.', lang_admin()));
        }
    }

    #[Layout('components.layouts.app', [
        'assets' => ['editor', 'tom-select'],
        'theme' => '7'
    ])]
    public function render()
    {
        return view('livewire.dashboard.company.general-information-form', [
            'title' => 'Edit general information',
            'page_title' => ['Company', 'General Information'],
        ]);
    }

    protected function rules()
    {
        $rules = [
            'data.company_name' => 'required|max:255',
            'data.company_ceo' => 'nullable|max:255',
            'data.company_introduction' => 'required|max:400',
            'data.company_description' => 'required',
            'data.year_of_established' => 'required|max:4',
            'data.number_of_employees' => 'nullable|max:255',
            'data.website' => 'nullable|max:500|url',
            'data.trademark' => 'required|max:255',
            'data.accepted_currencies' => 'required|array',
            'data.payment_terms' => 'required|array',
            'data.oem_services' => 'nullable|in:1,yes,no',
            'data.working_days' => 'required|array',
            'data.main_markets' => 'nullable|array',
        ];

        foreach (tab_langs() as $lang) {
            $suffix = $lang->status == 'default' ? '' : '_' . $lang->lang;
            $rules["data.company_name{$suffix}"] = $lang->status == 'default' ? 'required|max:255' : 'nullable|max:255';
            $rules["data.company_ceo{$suffix}"] = 'nullable|max:255';
            $rules["data.company_introduction{$suffix}"] = $lang->status == 'default' ? 'required|max:400' : 'nullable|max:400';
            $rules["data.company_description{$suffix}"] = $lang->status == 'default' ? 'required' : 'nullable';
        }

        return $rules;
    }
}