<?php

namespace App\Livewire\Dashboard\Company;

use App\Models\TradingInformation;
use App\Models\CompanyForm;
use App\Models\Country;
use App\Models\RfqForm;
use Livewire\Attributes\Layout;
use Livewire\Attributes\On;
use Livewire\Component;
use Illuminate\Support\Facades\DB;

class TradingInformationForm extends Component
{
    public $baseFields = [
        'import_export_license',
        'annual_turnover',
        'export_revenue',
        'export_percentage',
        'total_capitalization',
        'nearest_port',
        'terms_of_trades',
    ];
    public $annual_turnovers = [];
    public $export_revenues = [];
    public $total_capitalizations = [];
    public $trade_terms = [];
    public $countries = [];
    public $item;
    public $data = [];

    public function mount($id)
    {
        $this->item = TradingInformation::firstOrCreate(['user_id' => $id], ['user_id' => $id]);
        $this->loadData();
        $this->annual_turnovers = CompanyForm::where('type', 'annual_turnover')->pluck('title', 'title')->all();
        $this->export_revenues = CompanyForm::where('type', 'export_revenue')->pluck('title', 'title')->all();
        $this->total_capitalizations = CompanyForm::where('type', 'total_capitalization')->pluck('title', 'title')->all();
        $this->trade_terms = RfqForm::where('type', 'trade_terms')->pluck('title1', 'title1')->all();
        $this->countries = Country::orderBy('name')->pluck('name', 'id')->all();
    }

    protected function loadData()
    {
        foreach ($this->baseFields as $field) {
            if ($field == 'terms_of_trades') {
                $this->data[$field] = json_decode($this->item->$field, true) ?? [];
            } else {
                $this->data[$field] = $this->item->$field;
            }
        }
    }

    #[On('tomselect-updated')]
    public function handleTomSelectUpdated($field, $value)
    {
        try {
            $this->data[$field] = $value;
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: 'Failed to handle TomSelect update.');
        }
    }

    public function save()
    {

        try {
            $this->validate();
            DB::beginTransaction();

            $this->item->update([
                'import_export_license' => ($this->data['import_export_license'] == '1' || $this->data['import_export_license'] == 'yes') ? 'yes' : 'no',
                'annual_turnover' => $this->data['annual_turnover'],
                'export_revenue' => $this->data['export_revenue'],
                'export_percentage' => $this->data['export_percentage'],
                'total_capitalization' => $this->data['total_capitalization'],
                'nearest_port' => $this->data['nearest_port'],
                'terms_of_trades' => json_encode($this->data['terms_of_trades']),
            ]);

            DB::commit();

            $this->dispatch('insert', icon: 'success', title: 'Success', text: translateText('The form was submitted successfully.', lang_admin()));
            $this->loadData();
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            $messages = $e->validator->errors()->all();
            $errorMessage = implode('<br>', array_map(function ($error) {
                $error = str_replace('data.', '', $error);
                return "- $error";
            }, $messages));
            $this->dispatch('errorShow', icon: 'warning', title: '! Validation Error !', text: $errorMessage);
        } catch (\Exception $e) {
            DB::rollBack();
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: translateText('An unexpected error occurred.', lang_admin()));
        }
    }

    #[Layout('components.layouts.app', [
        'assets' => ['tom-select'],
        'theme' => '7'
    ])]
    public function render()
    {
        return view('livewire.dashboard.company.trading-information-form', [
            'title' => 'Edit Trading Information',
            'page_title' => ['Company', 'Trading Information'],
        ]);
    }

    protected function rules()
    {
        return [
            'data.import_export_license' => 'nullable|in:1,yes,no',
            'data.annual_turnover' => 'required|string|exists:company_form,title',
            'data.export_revenue' => 'required|string|exists:company_form,title',
            'data.export_percentage' => 'nullable|integer|min:0|max:100',
            'data.total_capitalization' => 'required|string|exists:company_form,title',
            'data.nearest_port' => 'required|exists:countries,id',
            'data.terms_of_trades' => 'required|array|min:1',
            'data.terms_of_trades.*' => 'string|exists:rfq_form,title1',
        ];
    }
}