<?php

namespace App\Livewire\Dashboard\ItemsAndServices\Product;

use Dflydev\DotAccessData\Data;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Livewire\Attributes\Layout;
use Livewire\Attributes\On;
use Livewire\Component;
use App\Models\Country;
use App\Models\Region;
use App\Models\Product;
use App\Models\Group;
use App\Models\RfqForm;
use App\Models\ProductPriceSummary;

use App\Http\Traits\StatusTrait;
use App\Http\Traits\Validation\ProductValidationTrait;

class Create extends Component
{
    use StatusTrait, ProductValidationTrait;

    public $baseFields = [
        //tab 1
        'slug',
        'category_id',
        'group_id',
        'us_hts_code',
        'brand',
        'model_number',
        'place_of_origin',
        'certifications',
        'other_certification',
        'gallery',
        'brochure',
        'video',
        'certificates',

        //tab 3
        'minimum_quantity',
        'minimum_unit',
        'fob_price',
        'ladder_prices',
        'amount',
        'max_amount',
        'negotiable',
        'main_market',
        'nearest_port',
        'payment_methods',
        'other_payment_methods',
        'sample_option',
        'lead_time_start',
        'lead_time_end',
        'sample_option_start',
        'sample_option_end',
        'sample_option_price',

        //tab 4
        'length',
        'width',
        'height',
        'weight',
        'showcase',
        'conditions',
    ];
    public $translatableFields = [
        //tab 1
        'title',
        'description',
        'keywords',
        'meta_title',
        'meta_description',
        'status',

        //tab 2
        'attributes',
    ];

    public $certifications = [
        "CE" => "CE",
        "FSC" => "FSC",
        "ROHS" => "ROHS",
        "CPC" => "CPC",
        "FCC" => "FCC",
        "UKCA" => "UKCA",
        "SEEDLING" => "SEEDLING",
        "GRS" => "GRS",
        "DIN CERTO" => "DIN CERTO",
        "CCC" => "CCC",
        "CB" => "CB",
        "BPI" => "BPI",
        "OK COMPOST HOME" => "OK COMPOST HOME",
        "OK BIOBASED" => "OK BIOBASED",
        "TUV MARK" => "TUV MARK",
        "EMC" => "EMC",
        "KC" => "KC",
        "IC" => "IC",
        "LVD" => "LVD",
        "CQC" => "CQC",
        "ALL" => "ALL",
    ];

    public $paymentMethods = [
        "CIA" => "CIA",
        "COD" => "COD",
        "CHECK" => "CHECK",
        "DA" => "DA",
        "DP" => "DP",
        "L/C" => "L/C",
        "Sight LC" => "Sight LC",
        "TT" => "TT",
        "ATT" => "ATT",
        "Paypal" => "Paypal",
        "WU" => "WU",
        "MG" => "MG",
    ];
    public $tools;
    public $countries;
    public $regions;
    public $groups = [];
    public $units = [];
    public $categoryTitle;
    public $trendingKeywords = [
        'Discount' => 'Discount',
        'New Arrival' => 'New Arrival',
        'Sale' => 'Sale',
        'Best Seller' => 'Best Seller',
        'Original' => 'Original',
        'Trendy' => 'Trendy',
        'Handmade' => 'Handmade',
        'Organic' => 'Organic',
        'Limited Edition' => 'Limited Edition',
        'Free Shipping' => 'Free Shipping',
        'High Quality' => 'High Quality',
        'Gift' => 'Gift',
        'Exclusive' => 'Exclusive',
        'Latest Model' => 'Latest Model',
        'Affordable' => 'Affordable',
        'Popular' => 'Popular',
        'Eco-Friendly' => 'Eco-Friendly',
        'Special Offer' => 'Special Offer',
        'Premium' => 'Premium',
        'Clearance' => 'Clearance',
    ];
    public $data = [];
    public $userId;

    public function someMethod()
    {

        return $this->tools;
    }

    public function mount($id)
    {
        $this->userId = $id;
        $this->loadData();
        $this->countries = Country::query()
            ->select('id', 'name', 'emoji', 'region')
            ->get()
            ->mapWithKeys(function ($country) {
                return [$country->id => "{$country->emoji} {$country->name} ({$country->region})"];
            })
            ->all();
        $this->regions = Region::pluck('name', 'id')->all();

        $this->units = RfqForm::where('type', 'units')->pluck('title1', 'title1')->all();
    }

    protected function loadData()
    {
        foreach ($this->baseFields as $field) {
            if (in_array($field, ['certifications', 'ladder_prices', 'payment_methods'])) {
                $this->data[$field] = [];
            } elseif (in_array($field, ['fob_price'])) {
                $this->data[$field] = 'fixed_price';
            } elseif (in_array($field, ['negotiable', 'showcase', 'conditions'])) {
                $this->data[$field] = false;
            } elseif (in_array($field, ['sample_option'])) {
                $this->data[$field] = 'no';
            } else {
                $this->data[$field] = null;
            }
        }
        foreach (tab_langs() as $lang) {
            $suffix = $lang->status == 'default' ? '' : '_' . $lang->lang;
            foreach ($this->translatableFields as $field) {
                $dbField = $field . $suffix;
                if (in_array($dbField, ['attributes'])) {
                    $this->data[$dbField] = [];
                } else {
                    $this->data[$dbField] = null;
                }

            }
        }
    }

    public function addAttribute($lang = null)
    {
        $suffix = ($lang && $lang !== tab_langs()->firstWhere('status', 'default')->lang)
            ? '_' . $lang
            : '';

        if (!isset($this->data['attributes' . $suffix])) {
            $this->data['attributes' . $suffix] = [];
        }

        $this->data['attributes' . $suffix][] = [
            'title' => '',
            'value' => ''
        ];
    }

    public function removeAttribute($index, $lang = null)
    {
        $suffix = ($lang && $lang !== tab_langs()->firstWhere('status', 'default')->lang)
            ? '_' . $lang
            : '';
        unset($this->data['attributes' . $suffix][$index]);
        $this->data['attributes' . $suffix] = array_values($this->data['attributes' . $suffix]);
    }

    private function fetchTrendingKeywords()
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer YOUR_API_KEY',
            ])->post('https://api.dataforseo.com/v3/keywords_data/google_trends/explore/live', [
                'keywords' => ['certificate', 'training', 'professional certification'],
                'location_name' => 'United States',
                'date_from' => 'today 1-m',
            ]);

            $this->trendingKeywords = $response->json()['tasks'][0]['result'] ?? [];
        } catch (\Exception $e) {
            $this->dispatch('alert', icon: 'error', title: 'Error', text: 'Failed to fetch trending keywords.');
        }
    }

    #[On('tomselect-updated')]
    public function handleTomSelectUpdated($field, $value)
    {
        try {
            $this->data[$field] = $value;
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: 'Failed to handle TomSelect update.');
        }
    }


    #[On('set-category-id')]
    public function setCategoryId($category_id, $category_title)
    {
        $this->data['category_id'] = $category_id;
        $this->categoryTitle = $category_title;
        $this->groups = Group::where('category_id', $category_id)->pluck('title', 'id')->all();
        $this->data['group_id'] = null;
        $this->dispatch('update-options-group_id', options: $this->groups);
    }

    public function save()
    {
        try {
            $this->validate();

            $this->status();
            $this->keywords();
            $this->attributes();

            DB::beginTransaction();
            if (isset($this->data['group_id']) && is_string($this->data['group_id'])) {
                $this->data['group_id'] = Group::create([
                    'category_id' => $this->data['category_id'][0],
                    'user_id' => $this->userId,
                    'title' => $this->data['group_id']
                ])->id;
            }

            $item = Product::create([
                'title' => $this->data['title'],
                'description' => $this->data['description'],
                'keywords' => $this->data['keywords'],
                'attributes' => $this->data['attributes'],
                'meta_title' => $this->data['meta_title'],
                'meta_description' => $this->data['meta_description'],
                'status' => $this->data['status'],
                'slug' => $this->data['slug'],
                'category_id' => $this->data['category_id'][0],
                'group_id' => $this->data['group_id'],
                'us_hts_code' => $this->data['us_hts_code'],
                'brand' => $this->data['brand'],
                'model_number' => $this->data['model_number'],
                'place_of_origin' => $this->data['place_of_origin'],
                'certifications' => json_encode($this->data['certifications']),
                'other_certification' => $this->data['other_certification'],
                'minimum_quantity' => $this->data['minimum_quantity'],
                'minimum_unit' => $this->data['minimum_unit'],
                'pricing_type' => $this->data['fob_price'],
                'negotiable' => $this->data['fob_price'] == 'negotiable' ? true : $this->data['negotiable'],
                'nearest_port' => $this->data['nearest_port'],
                'payment_methods' => json_encode($this->data['payment_methods']),
                'other_payment_methods' => $this->data['other_payment_methods'],
                'sample_option' => $this->data['sample_option'],
                'lead_time_start' => $this->data['lead_time_start'],
                'lead_time_end' => $this->data['lead_time_end'],
                'sample_option_start' => $this->data['sample_option_start'],
                'sample_option_end' => $this->data['sample_option_end'],
                'sample_option_price' => $this->data['sample_option_price'],
                'length' => $this->data['length'],
                'width' => $this->data['width'],
                'height' => $this->data['height'],
                'weight' => $this->data['weight'],
                'showcase' => $this->data['showcase'],
                'conditions' => $this->data['conditions'],
            ]);

            switch ($this->data['fob_price']) {
                case 'fixed_price':
                    $this->saveFixedPrice($item);
                    break;

                case 'ladder_price':
                    $this->saveLadderPrices($item);
                    break;

                case 'price_range':
                    $this->savePriceRange($item);
                    break;

                case 'negotiable':
                    break;
            }


            $this->updatePriceSummary($item);

            $item->main_markets()->sync($this->data['main_market']);

            store_lang_for_array_data($item, $this->data, ['title', 'description', 'keywords', 'meta_title', 'meta_description', 'status', 'attributes']);

            $this->dispatch('save-files', id: $item->id);

            DB::commit();

            $this->dispatch('insert', icon: 'success', title: 'success', text: translateText('The form was submitted successfully.', lang_admin()));
            $this->loadData();

        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            $messages = $e->validator->errors()->all();
            $errorMessage = implode('<br>', array_map(function ($error) {
                $error = str_replace('data.', '', $error);
                return "- $error";
            }, $messages));
            $this->dispatch('errorShow', icon: 'warning', title: '! Validation Error !', text: $errorMessage);
        } catch (\Exception $e) {
            DB::rollBack();
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: translateText('An unexpected error occurred.', lang_admin()));
        }
    }

    #[Layout('components.layouts.app', [
        'assets' => ['editor', 'tom-select', 'tom-enabling'],
        'theme' => '7'
    ])]
    public function render()
    {
        return view('livewire.dashboard.items-and-services.product.create', [
            'title' => 'Add Product',
            'page_title' => ['Items And Services', 'Product'],
        ]);
    }

    protected function rules()
    {
        return $this->productRules();
    }

    protected function messages()
    {
        return $this->productMessages();
    }

    protected function saveFixedPrice($product)
    {
        $product->prices()->create([
            'amount' => $this->data['amount'],
            'fixed_quantity' => null,
            'fixed_price' => null,
            'max_amount' => null
        ]);
    }

    protected function saveLadderPrices($product)
    {
        $product->prices()->delete();

        $ladderPrices = collect($this->data['ladder_prices'])->map(function ($price) use ($product) {
            return [
                'product_id' => $product->id,
                'fixed_quantity' => $price['fixed_quantity'],
                'fixed_price' => $price['fixed_price'],
                'amount' => null,
                'max_amount' => null,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        })->toArray();

        $product->prices()->insert($ladderPrices);
    }

    protected function savePriceRange($product)
    {
        $product->prices()->create([
            'amount' => $this->data['amount'],
            'max_amount' => $this->data['max_amount'],
            'fixed_quantity' => null,
            'fixed_price' => null
        ]);
    }

    protected function updatePriceSummary($product)
    {
        $minPrice = null;
        $maxPrice = null;

        switch ($product->pricing_type) {
            case 'fixed_price':
                $minPrice = $maxPrice = $product->prices->first()->amount;
                break;

            case 'ladder_price':
                $unitPrices = $product->prices->map(function ($price) {
                    return $price->fixed_price / $price->fixed_quantity;
                });
                $minPrice = $unitPrices->min();
                $maxPrice = $unitPrices->max();
                break;

            case 'price_range':
                $minPrice = $product->prices->first()->amount;
                $maxPrice = $product->prices->first()->max_amount;
                break;
        }

        ProductPriceSummary::updateOrCreate(
            ['product_id' => $product->id],
            [
                'min_unit_price' => $minPrice,
                'max_unit_price' => $maxPrice,
                'is_negotiable' => $product->negotiable
            ]
        );
    }
}

