<?php

namespace App\Livewire\Front\Rfq;

use App\Models\Banner;
use App\Models\ProductCategory;
use App\Models\Product;
use App\Models\RfqForm;
use App\Models\Rfq;
use App\Models\Photo;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithFileUploads;
use App\Http\Traits\AuthenticatesUser;
class RfqFormUser extends Component
{
    use WithFileUploads;
    use AuthenticatesUser;
    public $product_name = '';
    public $product_category = '';
    public $product_category_custom = '';
    public $required_quantity = '';
    public $select_unit = '';
    public $product_details = '';
    public $product_images = [];
    public $trade_terms = [];
    public $budget_limit = '';
    public $expected_unit_price = '';
    public $expected_unit_price_currency = '';
    public $buying_frequency = '';
    public $destination = '';
    public $shipping_method = [];
    public $payment_terms = '';
    public $delivery_time = '';
    public $rfq_expiry = '';
    public $rfq_expiry_type = '';
    public $banner;

    public function mount()
    {
        $this->banner = Banner::where('status', 'active')->where('place', 'rfq')->first();
        $this->ensureUserIsAuthenticated();
    }
    protected function rules()
    {
        return [
            'product_name' => 'required|string',
            'product_category' => 'nullable|exists:product_categories,id',
            'product_category_custom' => 'nullable|string',
            'required_quantity' => 'required|numeric|min:1',
            'select_unit' => 'required|string',
            'product_details' => 'required|string',
            'product_images' => 'required|array',
            'product_images.*' => 'required|mimes:jpg,png,docx,xlsx,pdf,rar,zip|max:6000',
            'trade_terms' => 'required|array',
            'trade_terms.*' => 'required|string',
            'budget_limit' => 'required|string',
            'expected_unit_price' => 'nullable|numeric',
            'expected_unit_price_currency' => 'nullable|string',
            'buying_frequency' => 'nullable|string',
            'destination' => 'required|string',
            'shipping_method' => 'required|array',
            'shipping_method.*' => 'required|string',
            'payment_terms' => 'required|string',
            'delivery_time' => 'nullable|numeric',
            'rfq_expiry' => 'nullable|numeric',
            'rfq_expiry_type' => 'nullable|string',
        ];
    }

    #[Computed]
    public function rfqForm()
    {
        return RfqForm::orderBy('sort')
            ->get();
    }

    #[Computed]
    public function grandchildren()
    {
        if (!$this->product_name) {
            return collect();
        }

        $queries = explode(',', $this->product_name);
        $lang_site = lang();

        $items = Product::whereHas('user_active')
            ->whereHas('c1_service_no')
            ->checkStatus();

        foreach ($queries as $query) {
            $query = trim($query);

            if ($lang_site == 'en') {
                $items = $items->orWhere('title', 'LIKE', "%{$query}%");
            } else {
                $items = $items->orWhereHas('langs', function ($lang) use ($query, $lang_site) {
                    $lang->where('lang', $lang_site)
                        ->where('col_name', 'title')
                        ->where('text', 'LIKE', "%{$query}%");
                });
            }
        }

        $items = $items->orderByDesc('id')->with(['category.children' => function ($query) {
            $query->checkStatus()->orderBy('sort');
        }])->get();


        return $items->map(function ($product) {
            return $product->category;
        })->unique('id');
    }

    public function submit()
    {
        $this->validate();

        try {
            $item = Rfq::create([
                'user_id' => auth()->id(),
                'product_name' => $this->product_name,
                'product_category_id' => $this->product_category,
                'product_category_custom' => $this->product_category_custom,
                'required_quantity' => $this->required_quantity,
                'select_unit' => $this->select_unit,
                'product_details' => $this->product_details,
                'trade_terms' => implode(' , ', $this->trade_terms),
                'budget_limit' => $this->budget_limit,
                'expected_unit_price' => $this->expected_unit_price,
                'expected_unit_price_currency' => $this->expected_unit_price_currency,
                'buying_frequency' => $this->buying_frequency,
                'destination' => $this->destination,
                'shipping_method' => implode(' , ', $this->shipping_method),
                'payment_terms' => $this->payment_terms,
                'delivery_time' => $this->delivery_time,
                'rfq_expiry' => $this->rfq_expiry,
                'rfq_expiry_type' => $this->rfq_expiry_type,
            ]);
            if ($this->product_images) {
                foreach ($this->product_images as $photo) {
                    $file = new Photo();
                    $file->type = 'file';

                    $originalName = $photo->getClientOriginalName();
                    $relativePath = '/assets/front/uploads/rfq-file/'.my_jdate(date('Y/m/d'), 'Y-m-d').'/'.time();
                    $photo->storeAs(
                        $relativePath,
                        $originalName,
                        'livewire_file'
                    );
                    $file->path = $relativePath.'/'.$originalName;
                    $item->file()->save($file);
                }
            }
            $this->dispatch('alert', icon: 'success', title: read_lang_word('msg-success-form'));
        } catch (\Exception $e) {
            $this->dispatch('alert', icon: 'error', title: read_lang_word('msg-err-form').$e->getMessage());
        }
    }

    #[Layout('front.layouts.front')]
    public function render()
    {
        return view('livewire.front.rfq.rfq-form-user', [
            'initialKeywords' => $this->product_name ? explode(',', $this->product_name) : []
        ]);
    }
}