<?php

namespace App\Http\Controllers\Admin\Gallery;

use App\Http\Controllers\Controller;
use App\Http\Requests\Gallery\GalleryRequest;
use App\Models\Gallery;
use App\Models\GalleryCategory;
use App\Models\Photo;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Spatie\Permission\Models\Permission;

class GalleryController extends Controller
{
    public function controller_title($type)
    {
        switch ($type) {
            case 'index':
                return translateText('گالری',lang_admin());
                break;
            case 'create':
                return translateText('افزودن گالری',lang_admin());
                break;
            case 'edit':
                return translateText('ویرایش گالری',lang_admin());
                break;
            case 'url_back':
                return route('admin.gallery.index');
                break;
            default:
                return '';
                break;
        }
    }

    public function __construct()
    {
        $this->middleware('permission:gallery_list', ['only' => ['index', 'show']]);
        $this->middleware('permission:gallery_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:gallery_edit', ['only' => ['edit', 'update', 'delete']]);
        $this->middleware('permission:gallery_delete', ['only' => ['destroy']]);
        $this->middleware('permission:gallery_status', ['only' => ['status']]);
        $this->middleware('permission:gallery_sort', ['only' => ['sort']]);
    }

    public function index()
    {
        if(auth()->user()->hasRole('supplier'))
            $items = GalleryCategory::where('user_id',auth()->id())->orderBy('sort')->get();
        else
        {
            $items=GalleryCategory::whereDoesntHave('user',function ($q){
                $q->role('supplier');
            })->orderBy('sort')->get();
        }
        return view('admin.gallery.index', compact('items'), ['title' => $this->controller_title('index')]);
    }

    public function show($id)
    {

    }

    public function create()
    {
        $url_back = $this->controller_title('url_back');
        return view('admin.gallery.create', compact('url_back'), ['title' => $this->controller_title('create')]);
    }

    public function store(GalleryRequest $request)
    {
        try {
            $item = GalleryCategory::create([
                'user_id' => auth()->id(),
                'title' => $request->title,
                'text' => $request->text,
                'status' => $request->status,
                'sort' => GalleryCategory::count() + 1,
            ]);
            //create pic
            if ($request->hasFile('photo')) {
                $photo = new Photo();
                $photo->type = 'photo';
                $photo->path = file_store($request->photo, 'assets/admin/uploads/gallery/photo' . '/');
                $item->photo()->save($photo);
            }
            // added gallery photos
            if ($request->hasFile('photos')) {
                foreach ($request->photos as $photo) {
                    $gallery = new Gallery();
                    $gallery->category_id = $item->id;
                    $gallery->type = 'photo';
                    $gallery->file = file_store($photo, 'assets/admin/uploads/gallery' . '/');
                    $gallery->save();
                }
            }
            // added gallery films
            if ($request->hasFile('films')) {
                foreach ($request->films as $film) {
                    $gallery = new Gallery();
                    $gallery->category_id = $item->id;
                    $gallery->type = 'video';
                    $gallery->file = file_store($film, 'assets/admin/uploads/gallery' . '/');
                    $gallery->save();
                }
            }

            store_lang($item, $request, ['title', 'text', 'status'], 'create');
            return redirect($this->controller_title('url_back'))->with('flash_message', translateText('اطلاعات با موفقیت افزوده شد',lang_admin()));
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', translateText('برای افزودن به مشکل خوردیم، مجدد تلاش کنید',lang_admin()));
        }
    }

    public function edit($id)
    {
        $url_back = $this->controller_title('url_back');

        if(auth()->user()->hasRole('supplier'))
            $item = GalleryCategory::where('user_id',auth()->id())->findOrFail($id);
        else
        {
            $item=GalleryCategory::whereDoesntHave('user',function ($q){
                $q->role('supplier');
            })->findOrFail($id);
        }
        return view('admin.gallery.edit', compact('url_back', 'item'), ['title' => $this->controller_title('edit')]);
    }

    public function update(GalleryRequest $request, $id)
    {
        if(auth()->user()->hasRole('supplier'))
            $item = GalleryCategory::where('user_id',auth()->id())->findOrFail($id);
        else
        {
            $item=GalleryCategory::whereDoesntHave('user',function ($q){
                $q->role('supplier');
            })->findOrFail($id);
        }
        try {
            $item->update([
                'title' => $request->title,
                'text' => $request->text,
                'status' => $request->status,
            ]);
            //edit pic
            if ($request->hasFile('photo')) {
                if ($item->photo) {
                    if (is_file($item->photo->path)) {
                        File::delete($item->photo->path);
                        delete_directory($item->photo->path);
                    }
                    $item->photo->delete();
                }
                $photo = new Photo();
                $photo->type = 'photo';
                $photo->path = file_store($request->photo, 'assets/admin/uploads/gallery/photo' . '/');
                $item->photo()->save($photo);
            }
//            added gallery photos
            if ($request->hasFile('photos')) {
                foreach ($request->photos as $photo) {
                    $gallery = new Gallery();
                    $gallery->category_id = $item->id;
                    $gallery->type = 'photo';
                    $gallery->file = file_store($photo, 'assets/admin/uploads/gallery' . '/');
                    $gallery->save();
                }
            }
            //            added gallery films
            if ($request->hasFile('films')) {
                foreach ($request->films as $film) {
                    $gallery = new Gallery();
                    $gallery->category_id = $item->id;
                    $gallery->type = 'video';
                    $gallery->file = file_store($film, 'assets/admin/uploads/gallery' . '/');
                    $gallery->save();
                }
            }
            store_lang($item, $request, ['title', 'text', 'status'], 'edit');
            return redirect($this->controller_title('url_back'))->with('flash_message',translateText('اطلاعات با موفقیت ویرایش شد',lang_admin()));
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', translateText('برای ویرایش به مشکل خوردیم، مجدد تلاش کنید',lang_admin()));
        }
    }

    public function destroy($id)
    {
        if(auth()->user()->hasRole('supplier'))
            $item = GalleryCategory::where('user_id',auth()->id())->findOrFail($id);
        else
        {
            $item=GalleryCategory::whereDoesntHave('user',function ($q){
                $q->role('supplier');
            })->findOrFail($id);
        }
        try {
            $item->delete();
            return redirect($this->controller_title('url_back'))->with('flash_message', translateText('اطلاعات با موفقیت حذف شد',lang_admin()));
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', translateText('برای حذف به مشکل خوردیم، مجدد تلاش کنید',lang_admin()));
        }
    }

    public function status($id, $type, $status)
    {
        if(auth()->user()->hasRole('supplier'))
            $item = GalleryCategory::where('user_id',auth()->id())->findOrFail($id);
        else
        {
            $item=GalleryCategory::whereDoesntHave('user',function ($q){
                $q->role('supplier');
            })->findOrFail($id);
        }
        try {
            $item->$type = $status;
            $item->update();
            return redirect($this->controller_title('url_back'))->with('flash_message', translateText('اطلاعات با موفقیت تغییر وضعیت شد',lang_admin()));
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', translateText('برای تغییر وضعیت به مشکل خوردیم، مجدد تلاش کنید',lang_admin()));
        }
    }

    public function sort($id, Request $request)
    {
        if(auth()->user()->hasRole('supplier'))
            $item = GalleryCategory::where('user_id',auth()->id())->findOrFail($id);
        else
        {
            $item=GalleryCategory::whereDoesntHave('user',function ($q){
                $q->role('supplier');
            })->findOrFail($id);
        }
        try {
            $item->sort = $request->sort;
            $item->update();
            return redirect($this->controller_title('url_back'))->with('flash_message', translateText('اطلاعات با موفقیت سورت شد',lang_admin()));
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', translateText('برای سورت به مشکل خوردیم، مجدد تلاش کنید',lang_admin()));
        }
    }

    public function delete($id)
    {
        if(auth()->user()->hasRole('supplier'))
        {
            $item = Gallery::whereHas('category',function ($q){
                $q->where('user_id',auth()->id());
            })->findOrFail($id);
        }
        else
        {
            $item = Gallery::whereHas('category',function ($q){
                $q->whereDoesntHave('user',function ($q){
                    $q->role('supplier');
                });
            })->findOrFail($id);
        }
        try {
            $item->delete();
            return redirect()->back()->with('flash_message', translateText('اطلاعات با موفقیت حذف شد',lang_admin()));
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', translateText('برای حذف به مشکل خوردیم، مجدد تلاش کنید',lang_admin()));
        }
    }
}
