<?php

namespace App\Http\Controllers\Admin\Product;

use App\Http\Controllers\Controller;
use App\Http\Requests\Product\CategoryRequest;
use App\Models\Photo;
use App\Imports\CatImport;
use App\Models\ProductCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Maatwebsite\Excel\Facades\Excel;

class CategoryController extends Controller
{
    public function controller_title($type)
    {
        switch ($type) {
            case 'index':
                return translateText('دسته بندی محصولات',lang_admin());
                break;
            case 'create':
                return translateText('افزودن دسته',lang_admin());
                break;
            case 'edit':
                return translateText('ویرایش دسته',lang_admin());
                break;
            case 'url_back':
                return route('admin.product-category.index');
                break;
            default:
                return '';
                break;
        }
    }

    public function __construct()
    {
        $this->middleware('permission:product_category_list', ['only' => ['index', 'show']]);
        $this->middleware('permission:product_category_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:product_category_edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:product_category_delete', ['only' => ['destroy']]);
        $this->middleware('permission:product_category_sort', ['only' => ['sort']]);
    }

    public function index()
    {
        $items = ProductCategory::with(['children.children'])
            ->where('parent_id', null)
            ->orderBy('sort')
            ->get();

        return view('admin.product.category.index', compact('items'), ['title' => $this->controller_title('index')]);
    }

    public function show($id)
    {

    }

    public function create()
    {
        $url_back = $this->controller_title('url_back');
        $parents = ProductCategory::where('parent_id', null)
            ->with(['children'])
            ->orderBy('sort')
            ->get();

        return view('admin.product.category.create', compact('url_back', 'parents'), ['title' => $this->controller_title('create')]);
    }

    public function store(CategoryRequest $request)
    {
        try {
            $item = ProductCategory::create([
                'parent_id' => $request->parent_id != 'father' ? $request->parent_id : null,
                'title' => $request->title,
                'status' => $request->status,
                'slug' => $request->slug
            ]);

            if ($request->hasFile('photo')) {
                $photo = new Photo();
                $photo->type = 'photo';
                $photo->path = file_store($request->photo, 'assets/admin/uploads/product-category' . '/');
                $item->photo()->save($photo);
            }
            if ($request->hasFile('icon')) {
                $photo = new Photo();
                $photo->type = 'icon';
                $photo->path = file_store($request->icon, 'assets/admin/uploads/product-category' . '/');
                $item->icon()->save($photo);
            }

            store_lang($item, $request, ['title', 'status'], 'create');

            return redirect($this->controller_title('url_back'))->with('flash_message', translateText('اطلاعات با موفقیت افزوده شد',lang_admin()));
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', translateText('برای افزودن به مشکل خوردیم، مجدد تلاش کنید',lang_admin()));
        }
    }

    public function edit($id)
    {
        $url_back = $this->controller_title('url_back');
        $item = ProductCategory::with(['parent'])->findOrFail($id);
        $parents = ProductCategory::where('parent_id', null)
            ->with(['children'])
            ->where('id', '!=', $id)
            ->orderBy('sort')
            ->get();

        return view('admin.product.category.edit', compact('url_back', 'item', 'parents'), ['title' => $this->controller_title('edit')]);
    }

    public function update(CategoryRequest $request, $id)
    {
        $item = ProductCategory::findOrFail($id);
        try {
            ProductCategory::where('id', $id)->update([
                'parent_id' => $request->parent_id != 'father' ? $request->parent_id : null,
                'title' => $request->title,
                'status' => $request->status,
                'slug' => $request->slug
            ]);
            //edit category
            if ($request->hasFile('photo')) {
                if ($item->photo) {
                    if (is_file($item->photo->path)) {
                        File::delete($item->photo->path);
                        delete_directory($item->photo->path);
                    }
                    $item->photo->delete();
                }
                $photo = new Photo();
                $photo->type = 'photo';
                $photo->path = file_store($request->photo, 'assets/admin/uploads/product-category' . '/');
                $item->photo()->save($photo);
            }
            //edit category
            if ($request->hasFile('icon')) {
                if ($item->icon) {
                    if (is_file($item->icon->path)) {
                        File::delete($item->icon->path);
                        delete_directory($item->icon->path);
                    }
                    $item->icon->delete();
                }
                $photo = new Photo();
                $photo->type = 'icon';
                $photo->path = file_store($request->icon, 'assets/admin/uploads/product-category' . '/');
                $item->icon()->save($photo);
            }

            store_lang($item, $request, ['title', 'status'], 'edit');

            return redirect($this->controller_title('url_back'))->with('flash_message', translateText('اطلاعات با موفقیت ویرایش شد',lang_admin()));
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', translateText('برای ویرایش به مشکل خوردیم، مجدد تلاش کنید',lang_admin()));
        }
    }

    public function destroy($id)
    {
        $item = ProductCategory::findOrFail($id);
        try {
            if(count($item->childs))
            {
                return redirect()->back()->withInput()->with('err_message', translateText('برای حذف به مشکل خوردیم، والد دسته های دیگر است',lang_admin()));
            }
            if(count($item->products))
            {
                return redirect()->back()->withInput()->with('err_message', translateText('برای حذف به مشکل خوردیم، به محصولات وصل است',lang_admin()));
            }
            $item->delete();
            return redirect($this->controller_title('url_back'))->with('flash_message', translateText('اطلاعات با موفقیت حذف شد',lang_admin()));
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', translateText('برای حذف به مشکل خوردیم، مجدد تلاش کنید',lang_admin()));
        }
    }

    public function sort($id, Request $request)
    {
        $item = ProductCategory::findOrFail($id);
        try {
            $item->sort = $request->sort;
            $item->update();
            return redirect($this->controller_title('url_back'))->with('flash_message', translateText('اطلاعات با موفقیت سورت شد',lang_admin()));
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', translateText('برای سورت به مشکل خوردیم، مجدد تلاش کنید',lang_admin()));
        }
    }
    public function import_excel(Request $request)
    {
        $this->validate($request, [
            'file' => 'required|mimes:xls,xlsx',
        ],
            [
                'file.required' => 'لطفا  فایل اکسل را وارد کنید',
                'file.mimes' => 'لطفا فرمت صحیح اکسل را انتخاب کنید(xls,xlsx)',
            ]);
        Excel::import(new CatImport, $request->file('file'));
        return redirect()->back()->with('flash_message', 'با موفقیت ثبت شد.');
    }

}
