<?php

namespace App\Http\Traits\Validation;

trait ProductValidationTrait
{
    protected function productRules()
    {
        $rules = [
            'data.slug' => 'required|max:255|unique:products,slug',
            'data.category_id' => 'required|exists:product_categories,id',
            'data.group_id' => 'nullable',
            'data.us_hts_code' => 'nullable|regex:/^\d{4}\.\d{2}\.\d{2}\.\d{2}$/',
            'data.brand' => 'nullable|max:255',
            'data.model_number' => 'nullable|numeric',
            'data.place_of_origin' => 'nullable|exists:countries,id',
            'data.certifications' => 'nullable|array',
            'data.other_certification' => 'nullable|max:255',

            'data.minimum_quantity' => 'required|numeric|min:1',
            'data.minimum_unit' => 'required|string|max:50',
            'data.fob_price' => 'required|in:fixed_price,ladder_price,price_range,negotiable',
            'data.ladder_prices' => 'required_if:data.fob_price,ladder_price|array',
            'data.ladder_prices.*.fixed_quantity' => 'required_if:data.fob_price,ladder_price|numeric|min:1',
            'data.ladder_prices.*.fixed_price' => 'required_if:data.fob_price,ladder_price|numeric|min:0',
            'data.amount' => 'required_if:data.fob_price,fixed_price,price_range|numeric|min:0',
            'data.max_amount' => 'required_if:data.fob_price,price_range|numeric|min:0|gte:data.amount',
            'data.negotiable' => 'nullable|boolean',
            'data.main_market' => 'nullable|array',
            'data.nearest_port' => 'nullable|exists:countries,id',
            'data.payment_methods' => 'nullable|array',
            'data.other_payment_methods' => 'nullable|max:255',
            'data.lead_time_start' => 'nullable|numeric|min:1|lte:data.lead_time_end',
            'data.lead_time_end' => 'nullable|numeric|min:1|gte:data.lead_time_start',
            'data.sample_option' => 'required|in:yes,no,contact',
            'data.sample_option_start' => $this->data['sample_option'] === 'yes'
                ? 'required|numeric|min:1|lte:data.sample_option_end'
                : 'nullable|numeric|min:1|lte:data.sample_option_end',
            'data.sample_option_end' => $this->data['sample_option'] === 'yes'
                ? 'required|numeric|min:1|gte:data.sample_option_start'
                : 'nullable|numeric|min:1|gte:data.sample_option_start',
            'data.sample_option_price' => $this->data['sample_option'] === 'yes'
                ? 'required|numeric|min:0'
                : 'nullable|numeric|min:0',

            'data.length' => 'nullable|numeric|min:0',
            'data.width' => 'nullable|numeric|min:0',
            'data.height' => 'nullable|numeric|min:0',
            'data.weight' => 'nullable|numeric|min:0',
            'data.showcase' => 'nullable|boolean',
            'data.conditions' => 'required|accepted',
        ];

        foreach (tab_langs() as $lang) {
            $suffix = $lang->status == 'default' ? '' : '_' . $lang->lang;

            $rules["data.title{$suffix}"] = $lang->status == 'default' ? 'required|max:255' : 'nullable|max:255';
            $rules["data.meta_title{$suffix}"] = 'nullable|max:255';
            $rules["data.meta_description{$suffix}"] = 'nullable|max:500';
            $rules["data.keywords{$suffix}"] = $lang->status == 'default' ? 'required|array' : 'nullable|array';
            $rules["data.description{$suffix}"] = $lang->status == 'default' ? 'required' : 'nullable';
            $rules["data.status{$suffix}"] = 'nullable|boolean';

            $rules["data.attributes{$suffix}"] = 'nullable|array';
            $rules["data.attributes{$suffix}.*.title"] = 'required_with:data.attributes'.$suffix.'|max:255';
            $rules["data.attributes{$suffix}.*.value"] = 'required_with:data.attributes'.$suffix.'|max:255';
        }

        return $rules;
    }

    protected function productMessages()
    {
        return [
            'data.slug.required' => translateText('The product slug is required.', lang_admin()),
            'data.slug.unique' => translateText('This slug is already in use. Please choose a different one.', lang_admin()),
            'data.slug.max' => translateText('The slug may not be greater than 255 characters.', lang_admin()),
            'data.category_id.required' => translateText('Please select a category.', lang_admin()),
            'data.category_id.exists' => translateText('The selected category is invalid.', lang_admin()),
            'data.us_hts_code.regex' => translateText('US HTS code must be in format 9999.99.99.99', lang_admin()),
            'data.brand.max' => translateText('The brand name may not be greater than 255 characters.', lang_admin()),
            'data.model_number.numeric' => translateText('The model number must be a number.', lang_admin()),
            'data.place_of_origin.exists' => translateText('The selected country of origin is invalid.', lang_admin()),
            'data.other_certification.max' => translateText('Other certification may not be greater than 255 characters.', lang_admin()),

            'data.minimum_quantity.required' => translateText('Minimum order quantity is required.', lang_admin()),
            'data.minimum_quantity.numeric' => translateText('Minimum order quantity must be a number.', lang_admin()),
            'data.minimum_quantity.min' => translateText('Minimum order quantity must be at least 1.', lang_admin()),
            'data.minimum_unit.required' => translateText('Minimum order unit is required.', lang_admin()),
            'data.minimum_unit.max' => translateText('Minimum order unit may not be greater than 50 characters.', lang_admin()),

            'data.fob_price.required' => translateText('Price type is required.', lang_admin()),
            'data.fob_price.in' => translateText('Selected price type is invalid.', lang_admin()),
            'data.ladder_prices.required_if' => translateText('Ladder prices are required when price type is ladder.', lang_admin()),
            'data.ladder_prices.*.fixed_quantity.required_if' => translateText('Quantity is required for ladder pricing.', lang_admin()),
            'data.ladder_prices.*.fixed_quantity.numeric' => translateText('Quantity must be a number.', lang_admin()),
            'data.ladder_prices.*.fixed_quantity.min' => translateText('Quantity must be at least 1.', lang_admin()),
            'data.ladder_prices.*.fixed_price.required_if' => translateText('Price is required for ladder pricing.', lang_admin()),
            'data.ladder_prices.*.fixed_price.numeric' => translateText('Price must be a number.', lang_admin()),
            'data.ladder_prices.*.fixed_price.min' => translateText('Price must be at least 0.', lang_admin()),
            'data.amount.required_if' => translateText('Price amount is required.', lang_admin()),
            'data.amount.numeric' => translateText('Price must be a number.', lang_admin()),
            'data.amount.min' => translateText('Price must be at least 0.', lang_admin()),
            'data.max_amount.required_if' => translateText('Maximum price is required for price range.', lang_admin()),
            'data.max_amount.numeric' => translateText('Maximum price must be a number.', lang_admin()),
            'data.max_amount.min' => translateText('Maximum price must be at least 0.', lang_admin()),
            'data.max_amount.gte' => translateText('Maximum price must be greater than or equal to minimum price.', lang_admin()),
            'data.negotiable.boolean' => translateText('Negotiable field must be true or false.', lang_admin()),

            'data.main_market.array' => translateText('Main markets must be an array.', lang_admin()),
            'data.nearest_port.exists' => translateText('The selected nearest port is invalid.', lang_admin()),
            'data.payment_methods.array' => translateText('Payment methods must be an array.', lang_admin()),
            'data.other_payment_methods.max' => translateText('Other payment method may not be greater than 255 characters.', lang_admin()),
            'data.lead_time_start.numeric' => translateText('Lead time start must be a number.', lang_admin()),
            'data.lead_time_start.min' => translateText('Lead time start must be at least 1.', lang_admin()),
            'data.lead_time_start.lte' => translateText('Lead time start must be less than or equal to end time.', lang_admin()),
            'data.lead_time_end.numeric' => translateText('Lead time end must be a number.', lang_admin()),
            'data.lead_time_end.min' => translateText('Lead time end must be at least 1.', lang_admin()),
            'data.lead_time_end.gte' => translateText('Lead time end must be greater than or equal to start time.', lang_admin()),
            'data.sample_option.required' => translateText('Sample option is required.', lang_admin()),
            'data.sample_option.in' => translateText('Selected sample option is invalid.', lang_admin()),
            'data.sample_option_start.required' => translateText('Sample start quantity is required when sample option is yes.', lang_admin()),
            'data.sample_option_start.numeric' => translateText('Sample start quantity must be a number.', lang_admin()),
            'data.sample_option_start.min' => translateText('Sample start quantity must be at least 1.', lang_admin()),
            'data.sample_option_start.lte' => translateText('Sample start quantity must be less than or equal to end quantity.', lang_admin()),
            'data.sample_option_end.required' => translateText('Sample end quantity is required when sample option is yes.', lang_admin()),
            'data.sample_option_end.numeric' => translateText('Sample end quantity must be a number.', lang_admin()),
            'data.sample_option_end.min' => translateText('Sample end quantity must be at least 1.', lang_admin()),
            'data.sample_option_end.gte' => translateText('Sample end quantity must be greater than or equal to start quantity.', lang_admin()),
            'data.sample_option_price.required' => translateText('Sample price is required when sample option is yes.', lang_admin()),
            'data.sample_option_price.numeric' => translateText('Sample price must be a number.', lang_admin()),
            'data.sample_option_price.min' => translateText('Sample price must be at least 0.', lang_admin()),

            'data.length.numeric' => translateText('Length must be a number.', lang_admin()),
            'data.length.min' => translateText('Length must be at least 0.', lang_admin()),
            'data.width.numeric' => translateText('Width must be a number.', lang_admin()),
            'data.width.min' => translateText('Width must be at least 0.', lang_admin()),
            'data.height.numeric' => translateText('Height must be a number.', lang_admin()),
            'data.height.min' => translateText('Height must be at least 0.', lang_admin()),
            'data.weight.numeric' => translateText('Weight must be a number.', lang_admin()),
            'data.weight.min' => translateText('Weight must be at least 0.', lang_admin()),
            'data.showcase.boolean' => translateText('Showcase field must be true or false.', lang_admin()),
            'data.conditions.required' => translateText('You must accept the terms and conditions.', lang_admin()),
            'data.conditions.accepted' => translateText('You must accept the terms and conditions.', lang_admin()),

            'data.title.required' => translateText('Product title is required for default language.', lang_admin()),
            'data.title.max' => translateText('Product title may not be greater than 255 characters.', lang_admin()),
            'data.meta_title.max' => translateText('Meta title may not be greater than 255 characters.', lang_admin()),
            'data.meta_description.max' => translateText('Meta description may not be greater than 500 characters.', lang_admin()),
            'data.keywords.required' => translateText('Keywords are required for default language.', lang_admin()),
            'data.keywords.array' => translateText('Keywords must be an array.', lang_admin()),
            'data.description.required' => translateText('Description is required for default language.', lang_admin()),
            'data.status.boolean' => translateText('Status must be true or false.', lang_admin()),
            'data.attributes.*.title.required_with' => translateText('Attribute name is required.', lang_admin()),
            'data.attributes.*.title.max' => translateText('Attribute name may not be greater than 255 characters.', lang_admin()),
            'data.attributes.*.value.required_with' => translateText('Attribute value is required.', lang_admin()),
            'data.attributes.*.value.max' => translateText('Attribute value may not be greater than 255 characters.', lang_admin()),
        ];
    }
}