<?php

namespace App\Livewire\Dashboard\Company;

use App\Models\BusinessInformation;
use App\Models\CompanyForm;
use App\Models\ProductCategory;
use Livewire\Component;
use Livewire\Attributes\On;
use Livewire\Attributes\Layout;

class BusinessInformationForm extends Component
{
    public $data = [];
    public $bussiness_types = [];
    public $categories = [];
    public $ownership_models = [];
    public $item;
    public $page_title;
    public $title;

    public function mount($dashboard_code)
    {

        try {
            $this->item = BusinessInformation::firstOrCreate(
                ['dashboard_code' => $dashboard_code],
                ['dashboard_code' => $dashboard_code]
            );

            $this->data = [
                'bussiness_type' => json_decode($this->item->bussiness_type, true) ?? [],
                'category_id' => $this->item->categories->pluck('id')->toArray(),
                'registration_number' => $this->item->registration_number,
                'ownership_model' => $this->item->ownership_model,
                'main_product_keywords' => json_decode($this->item->main_product_keywords, true) ?? [],
                'tax_identification_number' => $this->item->tax_identification_number,
            ];

            $this->bussiness_types = CompanyForm::where('type', 'bussiness_type')
                ->pluck('title', 'title')
                ->all();

            $this->categories = ProductCategory::where('parent_id', null)
                ->orderBy('sort')
                ->pluck('title', 'id')
                ->all();

            $this->ownership_models = CompanyForm::where('type', 'ownership_model')
                ->pluck('title', 'title')
                ->all();

            $this->page_title = ['Company', translateText('Business Information', lang_admin())];
            $this->title = translateText('Edit Business Information', lang_admin());
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: 'Failed to initialize form.');
        }
    }

    public function save()
    {
        try {
            $this->validate();

            $this->item->update([
                'bussiness_type' => json_encode($this->data['bussiness_type']),
                'registration_number' => $this->data['registration_number'],
                'ownership_model' => $this->data['ownership_model'],
                'main_product_keywords' => json_encode($this->data['main_product_keywords']),
                'tax_identification_number' => $this->data['tax_identification_number'],
            ]);
            $this->item->categories()->sync($this->data['category_id']);

            $this->dispatch('insert', icon: 'success', title: 'Success', text: translateText('The form was submitted successfully.', lang_admin()));
        } catch (\Illuminate\Validation\ValidationException $e) {
            $messages = $e->validator->errors()->all();
            $errorMessage = implode('<br>', array_map(function ($error) {
                $error = str_replace('data.', '', $error);
                return "- $error";
            }, $messages));
            $this->dispatch('errorShow', icon: 'warning', title: '! Validation Error !', text: $errorMessage);
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: 'Failed to save form.');
        }
    }

    #[On('tomselect-updated')]
    public function handleTomSelectUpdated($field, $value)
    {
        try {
            $this->data[$field] = $value;
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: 'Failed to handle TomSelect update.');
        }
    }

    protected function rules()
    {
        return [
            'data.bussiness_type' => 'required|array|min:1',
            'data.bussiness_type.*' => 'string|exists:company_form,title',
            'data.category_id' => 'required|array|min:1',
            'data.category_id.*' => 'exists:product_categories,id',
            'data.registration_number' => 'nullable|string|max:255',
            'data.ownership_model' => 'nullable|string|exists:company_form,title',
            'data.main_product_keywords' => 'required|array|min:1',
            'data.main_product_keywords.*' => 'string|max:255',
            'data.tax_identification_number' => 'nullable|string|max:255',
        ];
    }

    #[Layout('components.layouts.app', [
        'assets' => ['tom-select'],
        'theme' => '7'
    ])]
    public function render()
    {
        return view('livewire.dashboard.company.business-information-form');
    }
}