<?php

namespace App\Livewire\Dashboard\Company\CompanyOffice;

use App\Models\User;
use Livewire\Attributes\Layout;
use Livewire\Attributes\On;
use Livewire\Component;
use App\Models\OfficeInformation;
use App\Services\DataCacheService;
use App\Models\State;
use App\Models\City;

class Edit extends Component
{
    public $item;
    public $countries = [];
    public $states = [];
    public $cities = [];
    public $time_zone;
    public $baseFields =[
        'country_id',
        'state_id',
        'city_id',
        'street',
        'post_code',
        'time_zone',
    ];
    public $data = [];

    public function mount($id)
    {
        $this->item = OfficeInformation::with(['country', 'state', 'city'])
            ->where('id', $id)
            ->first();
        $this->authorize('update', $this->item);
        $this->loadData();
        $this->countries = DataCacheService::getCountries();
        $this->time_zone = DataCacheService::getTimeZones();
        $this->loadStatesAndCities();
    }

    protected function loadData()
    {
        try {
            foreach ($this->baseFields as $field) {
                $this->data[$field] = $this->item->$field;
            }
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: 'Failed to load form data.');
        }
    }

    protected function loadStatesAndCities()
    {
        try {
            $this->states = $this->data['country_id'] ? State::where('country_id', $this->data['country_id'])->pluck('name', 'id')->all() : [];
            $this->cities = $this->data['state_id'] ? City::where('state_id', $this->data['state_id'])->pluck('name', 'id')->all() : [];
            $this->dispatch('update-options-state_id', options: $this->states);
            $this->dispatch('update-options-city_id', options: $this->cities);
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: 'Failed to load states and cities.');
        }
    }

    #[On('tomselect-updated')]
    public function handleTomSelectUpdated($field, $value)
    {
        try {
            $this->data[$field] = $value;
            if ($field === 'country_id') {
                $this->data['state_id'] = null;
                $this->data['city_id'] = null;
                $this->states = $value ? State::where('country_id', $value)->pluck('name', 'id')->all() : [];
                $this->cities = [];
                $this->dispatch('update-options-state_id', options: $this->states);
                $this->dispatch('update-options-city_id', options: $this->cities);
            } elseif ($field === 'state_id') {
                $this->data['city_id'] = null;
                $this->cities = $value ? City::where('state_id', $value)->pluck('name', 'id')->all() : [];
                $this->dispatch('update-options-city_id', options: $this->cities);
            }
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: 'Failed to handle TomSelect update.');
        }
    }

    public function save()
    {
        $this->authorize('update', $this->item);
        try {
            $this->validate();
            $this->item->update([
                'country_id' => $this->data['country_id'],
                'state_id' => $this->data['state_id'],
                'city_id' => $this->data['city_id'],
                'street' => $this->data['street'],
                'post_code' => $this->data['post_code'],
                'time_zone' => $this->data['time_zone'],
            ]);
            $this->dispatch('insert', icon: 'success', title: 'Success', text: translateText('The form was submitted successfully.', lang_admin()));
        } catch (\Illuminate\Validation\ValidationException $e) {
            $messages = $e->validator->errors()->all();
            $errorMessage = implode('<br>', array_map(function ($error) {
                $error = str_replace('data.', '', $error);
                return "- $error";
            }, $messages));
            $this->dispatch('errorShow', icon: 'warning', title: '! Validation Error !', text: $errorMessage);
        } catch (\Exception $e) {
            $this->dispatch('errorShow', icon: 'error', title: 'Error', text: $e->getMessage());
        }
    }

    #[Layout('components.layouts.app', [
        'assets' => ['table', 'tom-select', 'country-manage'],
        'theme' => '7'
    ])]
    public function render()
    {
        return view('livewire.dashboard.company.company-office.edit', [
            'title' => 'Edit Company Office',
            'page_title' => ['Company', 'Company Office'],
        ]);
    }


    protected function rules()
    {
        $rules = [
            'data.country_id' => 'required|exists:countries,id',
            'data.state_id' => 'required|exists:states,id',
            'data.city_id' => 'required|exists:cities,id',
            'data.street' => 'required|string|max:255',
            'data.post_code' => 'required|string|max:20',
            'data.time_zone' => 'required|string|max:50',
        ];
        return $rules;
    }
}