<?php

namespace App\Livewire\Dashboard\UserDashboard;

use Livewire\Component;
use App\Models\{
    BusinessInformation,
    ContactInformation,
    FactoryInformation,
    GeneralInformation,
    QualityControl,
    TradingInformation,
    CompanyCertificate,
    CompanyFaq,
    OfficeInformation
};
use Illuminate\Support\Facades\Auth;

class ProfileCompletion extends Component
{
    public $completion = [
        'general' => 0,
        'contact' => 0,
        'business' => 0,
        'trading' => 0,
        'factory' => 0,
        'quality' => 0,
        'certificates' => 0,
        'faqs' => 0,
        'offices' => 0,
        'products' => 0,
    ];

    public $totalCompletion = 0;

    public function mount()
    {
        $this->calculateCompletion();
    }

    protected function calculateCompletion()
    {
        $user = Auth::user();
        $this->completion['general'] = $this->checkGeneralInfo($user);
        $this->completion['contact'] = $this->checkContactInfo($user);
        $this->completion['business'] = $this->checkBusinessInfo($user);
        $this->completion['trading'] = $this->checkTradingInfo($user);
        $this->completion['factory'] = $this->checkFactoryInfo($user);
        $this->completion['quality'] = $this->checkQualityControl($user);
        $this->completion['certificates'] = $this->checkCertificates($user);
        $this->completion['faqs'] = $this->checkFaqs($user);
        $this->completion['offices'] = $this->checkOffices($user);
        $this->completion['products'] = 0;

        $this->calculateTotal();
    }

    protected function isFieldFilled($value)
    {
        if (is_null($value) || (is_string($value) && trim($value) === '')) {
            return false;
        }
        if (is_array($value) && empty($value)) {
            return false;
        }
        return true;
    }

    protected function checkGeneralInfo($user)
    {
        $info = GeneralInformation::where('dashboard_code', $user->dashboard_code)->first();
        if (!$info) return 0;

        $requiredFields = [
            'year_of_established',
            'number_of_employees',
            'website',
            'accepted_currencies',
            'payment_terms',
            'oem_services',
            'working_days',
            'logo',
            'banner1',
            'banner2',
            'brochure',
            'video',
            'company_name',
            'company_ceo',
            'company_introduction',
            'company_description'
        ];
        $filled = 0;

        foreach ($requiredFields as $field) {
            if ($this->isFieldFilled($info->$field)) {
                $filled++;
            }
        }

        \Log::debug('General Info Fields', [
            'dashboard_code' => $user->id,
            'filled' => $filled,
            'total' => count($requiredFields),
            'percentage' => ($filled / count($requiredFields)) * 20
        ]);

        return ($filled / count($requiredFields)) * 20;
    }

    protected function checkBusinessInfo($user)
    {
        $info = BusinessInformation::where('dashboard_code', $user->dashboard_code)->first();
        if (!$info) return 0;

        $requiredFields = [
            'bussiness_type',
            'registration_number',
            'ownership_model',
            'main_product_keywords',
            'tax_identification_number',
        ];
        $filled = 0;

        foreach ($requiredFields as $field) {
            if ($this->isFieldFilled($info->$field)) {
                $filled++;
            } else {
                \Log::debug("Business Info: Field $field is empty", [
                    'dashboard_code' => $user->dashboard_code,
                    'value' => $info->$field
                ]);
            }
        }

        $percentage = ($filled / count($requiredFields)) * 10;
        \Log::debug('Business Info Completion', [
            'dashboard_code' => $user->dashboard_code,
            'filled' => $filled,
            'total' => count($requiredFields),
            'percentage' => $percentage
        ]);

        return $percentage;
    }

    protected function checkTradingInfo($user)
    {
        $info = TradingInformation::where('dashboard_code', $user->dashboard_code)->first();
        if (!$info) return 0;

        $requiredFields = [
            'import_export_license',
            'annual_turnover',
            'export_revenue',
            'export_percentage',
            'total_capitalization',
            'nearest_port',
            'terms_of_trades',
        ];
        $filled = 0;

        foreach ($requiredFields as $field) {
            if ($this->isFieldFilled($info->$field)) {
                $filled++;
            }
        }

        return ($filled / count($requiredFields)) * 10;
    }

    protected function checkFactoryInfo($user)
    {
        $info = FactoryInformation::where('dashboard_code', $user->dashboard_code)->first();
        if (!$info) return 0;

        $requiredFields = [
            'factory_location',
            'factory_total_area',
            'no_production_staff',
            'no_qc_staff',
            'no_rd_staff',
            'no_production_lines',
            'annual_output',
            'average_lead_time',
            'video',
            'gallery',
        ];
        $filled = 0;

        foreach ($requiredFields as $field) {
            if ($this->isFieldFilled($info->$field)) {
                $filled++;
            }
        }

        return ($filled / count($requiredFields)) * 5;
    }

    protected function checkQualityControl($user)
    {
        $info = QualityControl::where('dashboard_code', $user->dashboard_code)->first();
        if (!$info) return 0;

        $requiredFields = ['text'];
        $filled = 0;

        foreach ($requiredFields as $field) {
            if ($this->isFieldFilled($info->$field)) {
                $filled++;
            }
        }

        return ($filled / count($requiredFields)) * 5;
    }

    protected function checkContactInfo($user)
    {
        $info = ContactInformation::where('dashboard_code', $user->dashboard_code)->first();
        if (!$info) return 0;

        $requiredFields = [
            'email_address',
            'backup_email',
            'country_code1',
            'city_code',
            'phone_number',
            'country_code2',
            'mobile_number',
            'country_code3',
            'city_code2',
            'fax_number',
            'supported_languages',
            'social_media',
            'country_id',
            'state_id',
            'city_id',
            'street',
            'post_code',
            'time_zone',
            'profile',
            'contact_person',
            'job_title',
            'department',
        ];
        $filled = 0;

        foreach ($requiredFields as $field) {
            if ($this->isFieldFilled($info->$field)) {
                $filled++;
            }
        }

        return ($filled / count($requiredFields)) * 10;
    }

    protected function checkCertificates($user)
    {
        $count = CompanyCertificate::where('dashboard_code', $user->dashboard_code)->count();
        return $count > 0 ? 5 : 0;
    }

    protected function checkFaqs($user)
    {
        $count = CompanyFaq::where('dashboard_code', $user->dashboard_code)->count();
        return $count > 0 ? 5 : 0;
    }

    protected function checkOffices($user)
    {
        $office = OfficeInformation::where('dashboard_code', $user->dashboard_code)
            ->where('type', 'head')
            ->first();

        if (!$office) return 0;

        $requiredFields = [
            'country_id',
            'state_id',
            'city_id',
            'street',
            'post_code',
            'time_zone',
        ];
        $filled = 0;

        foreach ($requiredFields as $field) {
            if ($this->isFieldFilled($office->$field)) {
                $filled++;
            }
        }

        return ($filled / count($requiredFields)) * 5;
    }

    protected function calculateTotal()
    {
        $this->totalCompletion = round(array_sum($this->completion), 2);
    }
    public function refreshCompletion()
    {
        $this->calculateCompletion();
    }

    public function render()
    {
        return view('livewire.dashboard.user-dashboard.profile-completion');
    }
}